import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Camera, Heart, Users, Briefcase, Clock, Award, Star, User } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookSession.css'

const BookSession = () => {
  const navigate = useNavigate()
  const [selectedStyle, setSelectedStyle] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const styles = [
    { 
      id: 'portrait', 
      name: 'Портрет', 
      icon: Camera,
      description: 'Индивидуальные портретные съёмки',
      duration: '1-2 часа'
    },
    { 
      id: 'family', 
      name: 'Семейная', 
      icon: Users,
      description: 'Семейные фотосессии',
      duration: '2-3 часа'
    },
    { 
      id: 'lovestory', 
      name: 'Love Story', 
      icon: Heart,
      description: 'Романтические съёмки',
      duration: '2-4 часа'
    },
    { 
      id: 'brand', 
      name: 'Бренд', 
      icon: Briefcase,
      description: 'Брендовые фотосессии',
      duration: '3-5 часов'
    }
  ]

  const photographers = [
    { name: 'Анна Петрова', experience: '8 лет', rating: 5, specialty: 'Love Story, Портрет' },
    { name: 'Михаил Соколов', experience: '12 лет', rating: 5, specialty: 'Бренд, Портрет' },
    { name: 'Елена Волкова', experience: '6 лет', rating: 5, specialty: 'Семейная, Love Story' },
    { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Бренд, Портрет' }
  ]

  const features = [
    { icon: Award, title: 'Профессиональное оборудование', text: 'Топовое фотооборудование' },
    { icon: Clock, title: 'Гибкий график', text: 'Подстраиваемся под вас' },
    { icon: Star, title: 'Опытные фотографы', text: 'Многолетний опыт работы' },
    { icon: Camera, title: 'Быстрая обработка', text: 'Готовые фото за 3-5 дней' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedStyle && selectedDate) {
      const savedSessions = localStorage.getItem('photoSessions')
      const sessions = savedSessions ? JSON.parse(savedSessions) : []
      
      const styleData = styles.find(s => s.id === selectedStyle)
      const randomPhotographer = photographers[Math.floor(Math.random() * photographers.length)]
      
      const newSession = {
        id: Date.now(),
        photographer: randomPhotographer.name,
        theme: styleData.name,
        date: selectedDate,
        rating: 0
      }

      sessions.push(newSession)
      localStorage.setItem('photoSessions', JSON.stringify(sessions))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedStyle('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-session-page">
      <div className="container">
        <motion.div 
          className="book-header"
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <h1 className="book-title">Записаться на фотосессию</h1>
          <p className="book-subtitle">Выберите стиль съёмки и дату</p>
        </motion.div>

        <motion.section 
          className="booking-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.1 }}
        >
          <form className="booking-form" onSubmit={handleSubmit}>
            <div className="form-group">
              <label className="form-label">
                <Camera size={18} strokeWidth={2} />
                Стиль съёмки
              </label>
              <div className="styles-list">
                {styles.map((style) => {
                  const IconComponent = style.icon
                  return (
                    <motion.button
                      key={style.id}
                      type="button"
                      className={`style-item ${selectedStyle === style.id ? 'selected' : ''}`}
                      onClick={() => setSelectedStyle(style.id)}
                      whileHover={{ x: 5 }}
                      whileTap={{ scale: 0.98 }}
                    >
                      <div className="style-item-icon">
                        <IconComponent size={24} strokeWidth={2} />
                      </div>
                      <div className="style-item-content">
                        <h3 className="style-item-name">{style.name}</h3>
                        <p className="style-item-desc">{style.description}</p>
                        <span className="style-item-duration">{style.duration}</span>
                      </div>
                      {selectedStyle === style.id && (
                        <motion.div 
                          className="style-item-check"
                          initial={{ scale: 0 }}
                          animate={{ scale: 1 }}
                        >
                          <CheckCircle size={18} strokeWidth={2} />
                        </motion.div>
                      )}
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <div className="form-group">
              <label htmlFor="date" className="form-label">
                <Calendar size={18} strokeWidth={2} />
                Дата фотосессии
              </label>
              <input
                type="date"
                id="date"
                className="form-input"
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                min={minDate}
                max={maxDate}
                required
              />
            </div>

            <motion.button 
              type="submit" 
              className="submit-button" 
              disabled={!selectedStyle || !selectedDate}
              whileHover={{ y: -2 }}
              whileTap={{ scale: 0.98 }}
            >
              Записаться
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message"
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3 }}
              >
                <CheckCircle size={20} strokeWidth={2} />
                <span>Вы успешно записались на фотосессию!</span>
              </motion.div>
            )}
          </form>
        </motion.section>

        <motion.section 
          className="photographers-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <h2 className="section-title">Наши фотографы</h2>
          <div className="photographers-list">
            {photographers.map((photographer, index) => (
              <motion.div
                key={photographer.name}
                className="photographer-item"
                initial={{ opacity: 0, x: -30 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.1, duration: 0.4 }}
                whileHover={{ x: 8, borderColor: 'var(--primary)' }}
              >
                <div className="photographer-item-left">
                  <div className="photographer-item-icon">
                    <User size={24} strokeWidth={2} />
                  </div>
                  <div className="photographer-item-content">
                    <h3 className="photographer-item-name">{photographer.name}</h3>
                    <div className="photographer-item-details">
                      <span className="photographer-item-exp">{photographer.experience} опыта</span>
                      <div className="photographer-item-rating">
                        {Array.from({ length: 5 }, (_, i) => (
                          <Star key={i} size={14} fill="#A78BFA" stroke="#A78BFA" strokeWidth={1.5} />
                        ))}
                      </div>
                    </div>
                    <p className="photographer-item-specialty">{photographer.specialty}</p>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="features-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <h2 className="section-title">Преимущества</h2>
          <div className="features-grid">
            {features.map((feature, index) => {
              const IconComponent = feature.icon
              return (
                <motion.div
                  key={feature.title}
                  className="feature-item"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.4 }}
                  whileHover={{ x: 5, borderColor: 'var(--primary)' }}
                >
                  <div className="feature-item-icon">
                    <IconComponent size={24} strokeWidth={2} />
                  </div>
                  <div className="feature-item-content">
                    <h3 className="feature-item-title">{feature.title}</h3>
                    <p className="feature-item-text">{feature.text}</p>
                  </div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookSession
